/*****************************************************************************
**+------------------------------------------------------------------------+**
**|                                                                        |**
**|                Copyright 2010 Mistral Solutions Pvt Ltd.               |**
**|                                                                        |**
**|                                                                        |**
**|                                                                        |**   
**| This program is free software; you can redistribute it and/or          |**
**| modify it under the terms of the GNU General Public License as         |**
**| published by the Free Software Foundation; either version 2 of         |**
**| the License, or (at your option) any later version.                    |**
**|                                                                        |**
**| This program is distributed in the hope that it will be useful,        |**
**| but WITHOUT ANY WARRANTY; without even the implied warranty of         |**
**| MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the           |**
**| GNU General Public License for more details.                           |**
**|                                                                        |**      
**| You should have received a copy of the GNU General Public License      |**
**| along with this program; if not, write to the Free Software            |**
**| Foundation, Inc., 59 Temple Place, Suite 330, Boston,                  |**
**| MA 02111-1307 USA                                                      |**
**+------------------------------------------------------------------------+**
*****************************************************************************/   

/**
 * \file    ddr3_test.c
 *
 * \brief   Support routines for DDR test application
 *
 * This file contains the support function for DDR Test application.
 *
 * \author  0043
 *
 * \version 01a,13aug,2010 Created
 */

/*
 *====================
 * Includes
 *====================
 */
#include "stdio.h"
#include "DM814x_EVM.h"
#include "ddr3.h"

/*
 *====================
 * Static declarations
 *====================
 */
static UINT8 tx[DDR_RW_SIZE];
static UINT8 rx[DDR_RW_SIZE];

 /*
 *====================
 * Function implementations
 *====================
 */
 
STATUS ddr3_pattern_test 
(
	UINT32* pu32BaseAddr, 
	UINT32 u32Size, 
	UINT32 u32Pattern
)
{
	STATUS u32RetVal = SUCCESS;
	UINT32 u32Count = 0;
	UINT32* pu32Addr = pu32BaseAddr;
	
	/* ddr3 on EMIF 0 */
	for (u32Count = 0; u32Count < u32Size/4; u32Count++)
	{
		*pu32Addr++ = u32Pattern;		
	}
	
	/* Verify the pattern written to memory */
	pu32Addr = pu32BaseAddr;
	for (u32Count = 0; u32Count < u32Size/4; u32Count++)
	{
		if (*pu32Addr++ != u32Pattern)
		{
			printf ("Memory cell verification failed at addr 0x%x.\r\n",
					pu32Addr);
			u32RetVal = FAILED;
			break;
		}			
	}
	 
	 return (u32RetVal);
}
STATUS ddr3IncDatatest 
(
	UINT32* pu32BaseAddr, 
	UINT32 u32Size 
)
{
	STATUS u32RetVal = SUCCESS;
	UINT32 u32Count = 0;
	UINT32* pu32Addr = pu32BaseAddr;
	
	/* ddr3 on EMIF 0 */
	for (u32Count = 0; u32Count < u32Size/4; u32Count++)
	{
		*pu32Addr++ = u32Count;		
	}
	
	/* Verify the pattern written to memory */
	pu32Addr = pu32BaseAddr;
	for (u32Count = 0; u32Count < u32Size/4; u32Count++)
	{
		if (*pu32Addr++ != u32Count)
		{
			printf ("Incremental pattern Memory cell verification failed at addr 0x%x.\r\n",
					pu32Addr);
			u32RetVal = FAILED;
			break;
		}			
	}
	 
	 return (u32RetVal);
}

 /** **************************************************************************
 * \n \brief Routine to test ddr3 memory.
 *
 * This routine tests the ddr3 by writing into all banks and reading from it
 * then compares both .This routine takes  ddrctrl as parameter to select
 * the controller.
 * 
 * \param  ddrctrl   [IN]   To select controller
 *
 * \return
 * \n      return 0  for success  - Description
 * \n      return 1  for error   - Description
 */
INT16 ddr3_rw_test (INT16 ddrCtrl)
{
    INT16 i, j;
    UINT8* p8;
    UINT8 * src, *dst;
    UINT32 ddr_base, ddr_size, ddr_banks;

    if (DDR0_CTRL == ddrCtrl)
    {
		ddr_base = DDR0_BASE;
		ddr_size = DDR0_SIZE;
		ddr_banks = DDR0_BANK_NUM;
    }
    else if (DDR1_CTRL == ddrCtrl)
    {
		ddr_base = DDR1_BASE;
		ddr_size = DDR1_SIZE;
		ddr_banks = DDR1_BANK_NUM;
    }

    /* Write the known patterns to all banks of the DDR */
    for ( i = 0; i < ddr_banks; i++)
    {
        /* Create the test pattern */
        p8 = ( UINT8* )tx;
        for ( j = 0 ; j < DDR_RW_SIZE ; j++ )
            *p8++ = ( UINT8 )( j + i );

		/* Write the size of DDR_RW_SIZE to the bank of ddr3 memory */
		src = tx;
		dst = (UINT8 *) (ddr_base + i * (ddr_size/ddr_banks)); /* 16 MB */
		ddr3_write ((UINT16 *)src, (UINT16 *)dst, DDR_RW_SIZE);
    }

    /* Clear the receive buffer */
    for ( j = 0 ; j < DDR_RW_SIZE ; j++ )
        rx[j] = 0;

    /* Read and verify ddr3 */
    for ( i = 0 ; i < ddr_banks ; i++ )
    {
        /* Read a ddr3 bank */
		src = (UINT8 *)(ddr_base + i * (ddr_size/ddr_banks));
		//dst = rx;
        ddr3_read((UINT16 *)src, (UINT16 *)rx/*dst*/, DDR_RW_SIZE);

        /* Check the pattern */
        p8 = ( UINT8* )rx;
        for ( j = 0 ; j < DDR_RW_SIZE ; j++ )
            if ( ( *p8++ ) != ( UINT8 )( i + j ) )
                return 1;  // Fail
    }

    return 0;
}

/** **************************************************************************
 * \n \brief Routine to test the ddr3 with cache_enable option
 *
 * This routine takes cache_enable option as a parameter and does DDR read 
 * write tests.
 *
 * \return
 * \n      return SUCCESS for success  - Description
 * \n      return FAILED for error   - Description
 */
 
STATUS ddr3_test (INT16 cache_enable)
{
	STATUS u32RetVal = SUCCESS;
	
    if (cache_enable)
    {
        /* TODO - Check how to enable the cache */
	printf("ddr3_test: TODO - Enable the cache here\n");
    }
    else
    {
        /* TODO - Check how to disable the cache */
	printf("ddr3_test: TODO - Disable the cache here\n");
    }

    /* Initialize the DDR interface if any */
    ddr3_init( );
	
    /* Test the DDR connected to DDR0 */
    if (ddr3_rw_test(DDR0_CTRL))
	{
	    printf ("ddr3_test: Testing DDR connected to DDR0 failed !!!\n");
		u32RetVal = FAILED;
	}

    /* Test the DDR connected to DDR1 */
    if (ddr3_rw_test(DDR1_CTRL))
	{
		printf ("ddr3_test: Testing DDR connected to DDR1 failed !!!\n");
		u32RetVal = FAILED;
	}
#if 0
	
	/* testing the pattern write/readback */
	printf ("Carrying out 0xAAAAAAAA pattern test for DDR[0].\r\n");
	u32RetVal = ddr3_pattern_test ((UINT32*)DDR0_BASE, DDR0_SIZE, 0xAAAAAAAA);
	if (SUCCESS != u32RetVal)
	{
		printf ("Pattern 0xAAAAAAAA write Test failed for DDR0.\r\n");
	}
	printf ("Carrying out 0x55555555 pattern test for DDR[0].\r\n");
	u32RetVal = ddr3_pattern_test ((UINT32*)DDR0_BASE, DDR0_SIZE, 0x55555555);
	if (SUCCESS != u32RetVal)
	{
		printf ("Pattern 0x55555555 write Test failed for DDR0.\r\n");
	}
	printf ("Carrying out 0xAAAAAAAA pattern test for DDR[1].\r\n");
	u32RetVal = ddr3_pattern_test ((UINT32*)DDR1_BASE, DDR1_SIZE, 0xAAAAAAAA);
	if (SUCCESS != u32RetVal)
	{
		printf ("Pattern 0xAAAAAAAA write Test failed for DDR1.\r\n");
	}
	printf ("Carrying out 0x55555555 pattern test for DDR[1].\r\n");
	u32RetVal = ddr3_pattern_test ((UINT32*)DDR1_BASE, DDR1_SIZE, 0x55555555);
	if (SUCCESS != u32RetVal)
	{
		printf ("Pattern 0x55555555 write Test failed for DDR1.\r\n");
	}
#endif

	printf ("Carrying out Incremental pattern test for DDR[0].\r\n");
	u32RetVal = ddr3IncDatatest ((UINT32*)DDR0_BASE, DDR0_SIZE);
	if (SUCCESS != u32RetVal)
	{
		printf ("Incremental Pattern write Test failed for DDR[0].\r\n");
	}
	printf ("Carrying out Incremental pattern test for DDR[1].\r\n");
	u32RetVal = ddr3IncDatatest ((UINT32*)DDR1_BASE, DDR1_SIZE);
	if (SUCCESS != u32RetVal)
	{
		printf ("Incremental Pattern write Test failed for DDR[1].\r\n");
	}

    return u32RetVal;
}

